/**
 * This program is intended to showcase how a generic Linux USB driver can
 * interact with a Noritake GT-CP module.
 *
 * Created by: Noritake support
 * Date: 3/28/19
 *
 * NOTE: A generic Linux USB driver is intended for evaluation purposes.
 */

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>

// This may need to be changed to the correct
// GT-CP device path.
// Device path can be found using 'dmesg' in terminal.
#define GTCP_MODULE "/dev/ttyUSB0"

// Read and write USB device handles
int handle_for_read;
int handle_for_write;

// Global Module Type
// 0 = No type specified
// 1 = GT800X480A-C903PA
// 2 = GT480X272A-C903PA
int MODULE_TYPE = 0;

// Display drawing limitation Values
// Values specific for box drawing
int GT800X480A_H_LIMIT = 784;
int GT800X480A_V_LIMIT = 464;
int GT480X272A_H_LIMIT = 464;
int GT480X272A_V_LIMIT = 256;

// Draw box size
int BOX_SIZE = 15;

/**
 * Moves the cursor to a specific location on the GUTFT module.
 *
 * @param  x    The desired x coordinate.
 * @param  y    The desired y coordinate.
 * @return none
 */
void setCursor(int x, int y)
{
	char cursorCommand[] = {
		0x1f, '$', x, (x >> 8), y, (y >> 8)
	};
	write(handle_for_write, cursorCommand, 6);
}

/**
 * Clears the screen of the GUTFT module.
 *
 * @return none
 */
void clearScreen()
{
	char clearCommand[] = {
		0x0c
	};
	write(handle_for_write, clearCommand, 1);
}

/**
 * Enable or disable touch data transmission.
 *
 * enable = 0x00: Transmit OFF
 * enable = 0x01: Transmit ON
 * Default: enable = 0x00
 *
 * @param  enable Turn touch data ON or OFF.
 * @return none
 */
void enableTouch(char enable)
{
	char touchCommand[] = {
		0x1f, 0x50, 0x20, enable
	};
	write(handle_for_write, touchCommand, 4);
}

/**
 * Set the desired touch mode.
 *
 * numTouch = 0x00: Single touch mode
 * 0x01 <= numTouch <= 0x0a: Multi-touch mode (up to 10 touches)
 * Default: numTouch = 0x00
 *
 * @param  numTouch The maximum number of touches
 * @return none
 */
void multiTouchMode(char numTouch)
{
	char touchCommand[] = {
		0x1f, 0x50, 0x01, numTouch
	};
	write(handle_for_write, touchCommand, 4);
}

/**
 * Set the coordinate touch mode.
 *
 * 0x00 <= channel <= 0x03
 *
 * @param  channel The desired touch data channel.
 * @return none
 */
void coordinatesMode(char channel)
{
	char touchCommand[] = {
		0x1f, 0x50, 0x10, channel, 0x00
	};
	write(handle_for_write, touchCommand, 5);
}

/**
 * Draw a line or box on the display.
 *
 * mode = 0x00: Line
 * mode = 0x01: Box
 * mode = 0x02: Box fill
 *
 * pen = 0x00: Pen color is set to the background color.
 * pen = 0x01: Pen color is set to the character color.
 *
 * @param  mode   The desired drawing mode.
 * @param  pen    Set pen color to character or background color.
 * @param  xStart Line/Box drawing start position.
 * @param  yStart Line/Box drawing start position.
 * @param  xEnd   Line/Box drawing end position.
 * @param  yEnd   Line/Box drawing end position.
 * @return none
 */
void drawLineBox(char mode, char pen, unsigned xStart, unsigned yStart, unsigned xEnd, unsigned yEnd)
{
	char drawCommand[] = {
		0x1f, 0x28, 0x64, 0x11,
		mode, pen, xStart, xStart >> 8,
		yStart, yStart >> 8, xEnd, xEnd >> 8,
		yEnd, yEnd >> 8
	};
	write(handle_for_write, drawCommand, 14);
}

/**
 * Function to show where touches have occurred on the
 * display.
 * This function requires that coordinate mode is enabled
 * with multi-touch mode.
 *
 * @returns none
 */
void testTouch()
{
	char result[7];
	size_t readRet;

	// Set up touch parameters
	enableTouch(0x01); // Enable touch transmission
	multiTouchMode(10); // Set multi-touch mode for 10 max concurrent touches
	coordinatesMode(0x00); // Set coordinates mode to channel 0

	// Endless touch loop
	while(1)
	{
		// Read in one touch packet
		readRet = read(handle_for_read, result, 7);
		if(readRet > 0)
		{
			printf("Read Return Value: %ld\n", readRet);

			// Check if screen has been touched
			if(result[0] == 0x10 && result[1] == 0x11)
			{
				int xBox = ((result[4] & 0xff) << 8 | result[3] & 0xff);
				int yBox = ((result[6] & 0xff) << 8 | result[5] & 0xff);
				int xBoxEnd = xBox + BOX_SIZE;
				int yBoxEnd = yBox + BOX_SIZE;

				switch(MODULE_TYPE)
				{
					case 0: // No module selected
						// Do nothing
						break;
					case 1: // GT800X480A-C903PA selected
							// Make sure that boxes are not being drawn outside of
							// display area.
							if(xBox <= GT800X480A_H_LIMIT && yBox <= GT800X480A_V_LIMIT)
							{
								drawLineBox(0x02, 0x01, xBox, yBox, xBoxEnd, yBoxEnd);
							}
						break;
					case 2: // GT480X272A-C903PA selected
							// Make sure that boxes are not being drawn outside of
							// display area.
							if(xBox <= GT480X272A_H_LIMIT && yBox <= GT480X272A_V_LIMIT)
							{
								drawLineBox(0x02, 0x01, xBox, yBox, xBoxEnd, yBoxEnd);
							}
						break;
					default:
						// Do nothing
						break;
				}

				for(int i = 0; i < 7; i++)
				{
					printf(" %x\n", result[i]);
					result[i] = 0;
				}
			// Screen has been released
			} else if(result[0] == 0x10 && result[1] == 0x10)
			{
				printf("Release\n");
				for(int i = 0; i < 7; i++)
				{
					printf(" %x\n", result[i]);
					result[i] = 0;
				}
			}
		}
	}
}

// Helper function: Convert string characters to upper case.
void stringUpper(char *strg)
{
	while(*strg)
	{
		*strg = toupper(*strg);
		strg++;
	}
}

int main(int argc, char *argv[])
{
	//char *argument;
	//*argument = *argv[1];
	//printf(&argument);
	if(argc == 2)
	{
		stringUpper(argv[1]);
		printf("Argument %s\n", argv[1]);
		if(!strcmp(argv[1], "GT800X480A-C903PA"))
		{
			printf("7 Inch GT-CP Module\n");
			MODULE_TYPE = 1;
		} else if(!strcmp(argv[1], "GT480X272A-C903PA"))
		{
			printf("4.3 Inch GT-CP Module\n");
			MODULE_TYPE = 2;
		} else {
			printf("ERROR: Proper product number was not entered.\n");
			MODULE_TYPE = 0;
			exit(-1);
		}
	} else if(argc > 2)
	{
		printf("Too many arguments.\n");
		printf("Touch demo will not run.\n");
		exit(-1);
	} else if(argc < 2)
	{
		printf("Too few arguments.\n");
		printf("Touch demo will not run.\n");
		exit(-1);
	} else {
		printf("Unknown error.\n");
		exit(-1);
	}

	// Get read handle
	handle_for_read = open(GTCP_MODULE, O_RDONLY | O_NOCTTY);
	if(handle_for_read < 0)
	{
		perror(GTCP_MODULE);
		exit(-1);
	}

	// Get write handle
	handle_for_write = open(GTCP_MODULE, O_WRONLY | O_NOCTTY | O_NONBLOCK);
	if(handle_for_write < 0)
	{
		perror(GTCP_MODULE);
		exit(-1);
	}

	clearScreen();
	write(handle_for_write, "Hello World!", 12);

	testTouch();

	close(handle_for_read);
	close(handle_for_write);

	return 0;
}
