#include <Servo.h>
 
//ピン接続 | Pin assign
#define GT_DTR 4      //DTR
#define GT_DSR 6      //DSR
#define GT_TRDY 7      //TRDY <--未使用 | unused in this sample
 
#define RX_PIN 13 // GT-SPからのシリアルデータを受け取るピン | Pin to receive serial data from GT-S
 
Servo servoMotor;

String currentPage;
long RAD_min;
long RAD_max;

void setup() {
  //ピン初期設定 | Pin setting
  pinMode(GT_DTR, INPUT);
  pinMode(GT_DSR, OUTPUT);
  pinMode(GT_TRDY, INPUT);
 
  // Servoモータの初期化
  servoMotor.attach(RX_PIN);
 
  // シリアル通信の開始 | Start serial communication
  Serial.begin(38400);    //38400bps Baud rate
  Serial.setTimeout(100); //シリアルタイムアウト | Serial timeout

  // 開始ページの初期化 | Initialize starting page
  currentPage = "PAGE_0";

  //サーボモーター角度の初期化 | Initialize servo motor angles
  RAD_min = 0;
  RAD_max = 180;

}

void loop() {

  //GT-SPデータ受信 | Receive one GT-SP response data
  if ( Serial.available() ) {
    String receivedData = gtsp_signal_read();
    //Serial.println(receivedData); // 受け取ったデータをシリアルモニタに表示 | Display received data on serial monitor

    if (receivedData.startsWith("PAGE")){
      //ページ移動だった場合、現在ページ変数を更新 | Update current page variable if page navigation
      currentPage = receivedData;
    }else{
      //ページ移動ではない場合、現在のページごとの処理に飛ぶ | Jump to the current page's specific processing if not page navigation
      if (currentPage == "PAGE_0") {
          processPAGE_0(receivedData);
      }else if  (currentPage == "PAGE_1") {
          processPAGE_1(receivedData);
      }
    }
  }
}

////////////////////////////////////////////////////
//PAGE_0の処理 | Process - PAGE_0
////////////////////////////////////////////////////
void processPAGE_0(String data){
  //スライダー操作処理 | Slider operation processing
  if (data.startsWith("RAD")){
    //スライダーの値を取得 | Get slider value
    unsigned long gtspValue = gtsp_signal_read_numeric();
    //Serial.println(gtspValue); // 受け取ったデータをシリアルモニタに表示 | Display received data on serial monitor

    // スライダーの値をRAD_min〜RAD_maxの範囲にマッピング | Mapping slider values to the range of RAD_min to RAD_max
    int servoAngle = map(gtspValue, 0, 1023, RAD_min, RAD_max);

    // サーボモータを指定した角度に動かす | Move servo motor to the specified angle
    servoMotor.write(servoAngle);

    // GT-SPにサーボモータの角度を表示 | Display servo motor angle on GT-SP
    gtsp_ObjPrpSet_string(2, 0x40, String(servoAngle)+ " deg");

  }
}

////////////////////////////////////////////////////
//PAGE_1の処理 | Process - PAGE_1
////////////////////////////////////////////////////
void processPAGE_1(String data){
  //設定値の取得 | Retrieval of settings
  if (data.startsWith("RAD_SET")){
      RAD_min = gtsp_signal_read().toInt(); //最小値の取得 | Retrieve minimum value
      RAD_max = gtsp_signal_read().toInt(); //最大値の取得 | Retrieve maximum value
  }

  //設定値のエラーチェックと更新 | Check and update settings for errors
  if (RAD_min >= 0 && RAD_max <=180 && RAD_min < RAD_max ){
    // PAGTE_0 スライダー横のMIN MAX数値を更新 | Update MIN MAX numbers next to the slider on PAGE_0
    gtsp_ObjPrpSet_string(3, 0x40, String(RAD_min));
    gtsp_ObjPrpSet_string(4, 0x40, String(RAD_max));
  }else{
    //値の設定値が正しくないときは、エラーダイアログをPOPUPで表示 | Display error dialog if the settings values are incorrect
    gtsp_ObjPrpSet_string(13, 0x40, "RAD Value Error"); //エラーメッセージの変更 | Change error message
    gtsp_PgControl(0x20, 2); //エラーダイアログPOPUP表示 | Display error dialog POPUP
  }
}


/********************** 関数 | Function **********************/
////////////////////////////////////////////////////
//  データ受信（文字列用） | Receive data from GT-SP (for String)
////////////////////////////////////////////////////
String gtsp_signal_read(){

  byte res_dl[4] = "";
  unsigned long dl;
  char res_data_char[255]="";

  if (Serial.find("RESb", 4)){
    Serial.readBytes(res_dl, 4); //データ長抽出 | Read data length
    dl = (unsigned long)(res_dl[0] + (res_dl[1]<<8) + (res_dl[2]<<16) + (res_dl[3]<<24)); //データ長変換 | Data length
    Serial.readBytes(res_data_char, dl); //データ抽出 | Read data
    return String(res_data_char); //String型変換、リターン | Return as String
  }
}

////////////////////////////////////////////////////
//  データ受信（数値用） | Receive data from GT-SP (for Numeric)
////////////////////////////////////////////////////
unsigned long gtsp_signal_read_numeric() {
  byte res_dl[4] = "";
  unsigned long dl;
 
  if (Serial.find("RESb", 4)) {
    Serial.readBytes(res_dl, 4); //データ長抽出 | Read data length
    dl = (unsigned long)(res_dl[0] + (res_dl[1] << 8) + (res_dl[2] << 16) + (res_dl[3] << 24)); //データ長変換 | Data length
 
    // 数値を受信するためのバッファ
    byte res_data_buffer[4];
    // データを読み取り
    Serial.readBytes(res_data_buffer, dl);
 
    // 数値に変換してリターン
    return (unsigned long)(res_data_buffer[0] + (res_data_buffer[1] << 8) + (res_data_buffer[2] << 16) + (res_data_buffer[3] << 24));
  }
 
  // エラー時は0を返す（適切なエラー処理が必要な場合は修正が必要）
  return 0;
}
 
 
 
 
////////////////////////////////////////////////////
//  オブジェクト制御コマンド - プロパティ設定 (文字列用) | Object Control Command -Property Setting (for String)
////////////////////////////////////////////////////
void gtsp_ObjPrpSet_string(int obj, int prp, String val )
{
  gt_print("CMD");     //コマンドヘッダ | Command header
  gt_put(0xd3);        //オブジェクト-プロパティ設定コマンド | Object-Property Setting
  gt_put(obj >> 0);    //オブジェクトNo. 下位バイト | Object No. Lower byte
  gt_put(obj >> 8);    //オブジェクトNo. 上位バイト| Object No. Upper byte
  gt_put(prp >> 0);    //プロパティNo. 下位バイト | Property No. Lower byte 
  gt_put(prp >> 8);    //プロパティNo. 上位バイト| Property No. Upper byte 
  gt_put(val.length() >> 0);    //データ長 最下位バイト | Data length Least significant byte
  gt_put(val.length() >> 8);    //データ長 下位バイト | Data length second byte
  gt_put(val.length() >> 16);   //データ長 上位バイト | Data length third byte
  gt_put(val.length() >> 24);   //データ長 最上位バイト| Data length Most significant byte
  gt_print(val);       //文字列送信 // Send string
}

////////////////////////////////////////////////////
//  GUIページ制御/POPUP制御 | GUI PAGE/POPUP Control Command
////////////////////////////////////////////////////
void gtsp_PgControl(int typ, int pg)
{
  gt_print("CMD");     //コマンドヘッダ | Command header
  gt_put(0xfb);        //UIページ制御/POPUP制御設定コマンド | Object-Property Setting
  gt_put(typ);       //制御タイプ | Type (10h:PAGE Select, 20h:POPUP Open, 21h, POPUP Close)
  gt_put(pg >> 0);    //PAGE No. 下位バイト | Page No. Lower byte
  gt_put(pg >> 8);    //PAGE No. 上位バイト| Page No. Upper byte
}
 
////////////////////////////////////////////////////
//  1byte送信 | Send byte to GT-SP
////////////////////////////////////////////////////
void gt_put(unsigned char onebyte)
{
  while ( digitalRead(GT_DTR) == HIGH ) {}  //busycheck
  Serial.write(onebyte);
}
 
////////////////////////////////////////////////////
//  文字列送信 | Send String to GT-SP
////////////////////////////////////////////////////
void gt_print(String val)
{
  int val_i;
  //文字列を1文字ずつ送信 | Send string per one byte
  for (val_i = 0; val_i < val.length(); val_i++)
  {
    while ( digitalRead(GT_DTR) == HIGH ) {}  //busycheck
    Serial.print(val.substring(val_i, val_i+1));
  }
}  
