/*********************************************************
 * Filename: GT1P.c
 * 
 * Purpose:
 * Provide functions directly relating to the GT-1P's
 * commands to be used via USB connection on a Linux
 * platform. 
 * 
 * Version: 3.0 (ME-N63-01)
 * Date: 9/19/2018
 * Website: www.noritake-elec.com
 * (C) 2019 Noritake Co., Inc.
 *
 * Disclaimer: This software example is provided "AS-IS". We are not
 * responsible for any issues that may occur while using this program.
 * This software is intended for demonstration and evaluation purposes
 * only.
 *********************************************************/

#include <libusb-1.0/libusb.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h> // for sleep()

#include "GT1P_USB_LIB.h"
#include "GT1P.h"

#define BULK_TRANSFER_MAX 64
#define INTERRUPT_TRANSFER_MAX 18

/**
 * Requests boot information from the GT-1P module.
 * 
 * Returned data:
 * 1. Header : 0x28 : 1 byte
 * 2. Identifier1 : 0x65 : 1 byte
 * 3. Identifier2 : 0x40 : 1 byte
 * 4. Data : 0x00 - 0xff : 4 bytes
 * 
 * @param  data Array to store data from the module.
 * @return none
 */
void GT1P_productStatusSendBoot(unsigned char * bootData)
{
	int count = 0;

	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x40, 0x01
	};
	write_data(command, 5);
	
	while(7 != read_USB(bootData, BULK_TRANSFER_MAX))
	{
		count++;
		if(count > 5)
		{
			printf("USB read error.\nPlease reconnect your module.\n");
			return;
		}
	}
}

/**
 * Requests firmware version information from the GT-1P module.
 * 
 * Returned data:
 * 1. Header : 0x28 : 1 byte
 * 2. Identifier1 : 0x65 : 1 byte
 * 3. Identifier2 : 0x40 : 1 byte
 * 4. Data : 0x00 - 0xff : 4 bytes
 * 
 * @param  data Array to store data from the module.
 * @return none
 */
void GT1P_productStatusSendFirm(unsigned char * firmData)
{
	int count = 0;
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x40, 0x02
	};
	write_data(command, 5);
	
	while(7 != read_USB(firmData, BULK_TRANSFER_MAX))
	{
		count++;
		if(count > 5)
		{
			printf("USB read error.\nPlease reconnect your module.\n");
			return;
		}
	}
}

/**
 * Requests memory checksum information from the GT-1P module.
 * 
 * 0x00 <= address <= 0xff: Start address (Effective address = address * 0x10000)
 * 0x01 <= length <= 0xff: Data length (Effective data length = length * 0x10000)
 * 
 * Returned data:
 * 1. Header : 0x28 : 1 byte
 * 2. Identifier1 : 0x65 : 1 byte
 * 3. Identifier2 : 0x40 : 1 byte
 * 4. Data : 0x00 - 0xff : 4 bytes
 * 
 * @param  address The desired start address.
 * @param  length  The length of the memory check.
 * @param  data    Array to store data from the module.
 * @return none
 */
void GT1P_productStatusSendMemCheck(char address, char length, unsigned char * memCheckData)
{
	int count = 0;
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x40, 0x20, address, length
	};
	write_data(command, 7);
		
	while(7 != read_USB(memCheckData, BULK_TRANSFER_MAX))
	{
		count++;
		if(count > 5)
		{
			printf("USB read error.\nPlease reconnect your module.\n");
			return;
		}
	}
}

/**
 * Requests product type information from the GT-1P module.
 * 
 * Returned data:
 * 1. Header : 0x28 : 1 byte
 * 2. Identifier1 : 0x65 : 1 byte
 * 3. Identifier2 : 0x40 : 1 byte
 * 4. Data : 0x00 - 0xff : 15 bytes
 * 
 * @param  data Array to store data from the module.
 * @return none
 */
void GT1P_productStatusSendProdType(unsigned char * prodData)
{
	int count = 0;
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x40, 0x30
	};
	write_data(command, 5);
		
	while(18 != read_USB(prodData, BULK_TRANSFER_MAX))
	{
		count++;
		if(count > 5)
		{
			printf("USB read error.\nPlease reconnect your module.\n");
			return;
		}
	}
}

/**
 * Requests display X pixel information (width in pixels) from the
 * GT-1P module.
 * 
 * Returned data:
 * 1. Header : 0x28 : 1 byte
 * 2. Identifier1 : 0x65 : 1 byte
 * 3. Identifier2 : 0x40 : 1 byte
 * 4. Data : 0x00 - 0xff : 3 bytes
 * 
 * @param  data Array to store data from the module.
 * @return none
 */
void GT1P_productStatusSendWidth(unsigned char * widthData)
{
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x40, 0x40
	};
	write_data(command, 5);
	
	read_USB(widthData, BULK_TRANSFER_MAX);
}

/**
 * Requests display Y pixel information (height in pixels) from the
 * GT-1P module.
 * 
 * Returned data:
 * 1. Header : 0x28 : 1 byte
 * 2. Identifier1 : 0x65 : 1 byte
 * 3. Identifier2 : 0x40 : 1 byte
 * 4. Data : 0x00 - 0xff : 3 bytes
 * 
 * @param  data Array to store data from the module.
 * @return none
 */
void GT1P_productStatusSendHeight(unsigned char * heightData)
{
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x40, 0x41
	};
	write_data(command, 5);
	
	read_USB(heightData, BULK_TRANSFER_MAX);
}

/**
 * Requests the display's touch setting package name.
 * 
 * Only compatible with GT800X480A-1303P.
 * 
 * Returned data
 * 1. Header : 0x28 : 1 byte
 * 2. Identifier1 : 0x65 : 1 byte
 * 3. Identifier2 : 0x40 : 1 byte
 * 4. Data : 0x00 - 0xff : 15 bytes
 * 
 * @param  data Array to store data from the module.
 * @return none
 */
void GT1P_productStatusSendTSPName(unsigned char * TSPNameData)
{
	int count = 0;
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x40, 0x70
	};
	write_data(command, 5);
		
	while(18 != read_USB(TSPNameData, BULK_TRANSFER_MAX))
	{
		count++;
		if(count > 5)
		{
			printf("USB read error.\nPlease reconnect your module.\n");
			return;
		}
	}
}

/**
 * Requests the display's touch setting package ID.
 * 
 * Only compatible with GT800X480A-1303P.
 * 
 * Returned data
 * 1. Header : 0x28 : 1 byte
 * 2. Identifier1 : 0x65 : 1 byte
 * 3. Identifier2 : 0x40 : 1 byte
 * 4. Data : 0x00 - 0xff : 4 bytes
 * 
 * @param  data Array to store data from the module.
 * @return none
 */
void GT1P_productStatusSendTSPID(unsigned char * TSPIDData)
{
	int count = 0;
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x40, 0x71
	};
	write_data(command, 5);
		
	while(7 != read_USB(TSPIDData, BULK_TRANSFER_MAX))
	{
		count++;
		if(count > 5)
		{
			printf("USB read error.\nPlease reconnect your module.\n");
			return;
		}
	}
}

/**
 * Sets the display's brightness.
 * 
 * @param  level The desired brightness level.
 * @return none
 */
void GT1P_setBrightness(char level){
	unsigned char command[] = {
		0x1f, 0x58, level
	};
	write_data(command, 3);
}

/**
 * Sets the touch panel's threshold level.
 * 
 * @param  level The desired threshold level.
 * @return none
 */
void GT1P_setThreshold(char level){
	unsigned char command[] = {
		0x1f, 0x4b, 0x70, 0x00, level
	};
	write_data(command, 5);
}

/**
 * Sets the module's touch standard reference setting.
 * 
 * 0x00 <= value <= 0xff: Maximum noise value
 * 0x00 <= numbers <= 0xff: Designation of measurement numbers
 * 
 * Returned Data:
 * 1. Status : 0x00 - 0x02 : 1 byte
 * 2. NNh / XXh : 0x00 - 0xff : 1 byte
 * 3. NXh / XXh : 0x00 - 0xff : 1 byte
 * 4. NYh / XXh : 0x00 - 0xff : 1 byte
 * 
 * @param  value    The maximum noise value.
 * @param  numbers  The designation of measurement numbers.
 * @return none
 */
void GT1P_setTSRSetting(char value, char numbers, unsigned char * data)
{
	int count = 0;
	unsigned char command[] = {
		0x1f, 0x4b, 0x70, 0x06, value, numbers
	};
	write_data(command, 6);
	
	sleep(2); // let module process test
		
	
	while(4 != read_USB(data, BULK_TRANSFER_MAX))
	{
		count++;
		if(count > 5)
		{
			printf("USB read error.\nPlease reconnect your module.\n");
			return;
		}
	}
	
}

/**
 * Enables or disables the touch standard reference usage.
 * 
 * @param  enable  Turn touch standard reference usage ON or OFF.
 * @return none
 */
void GT1P_setTSRUsage(char enable)
{
	unsigned char command[] = {
		0x1f, 0x4b, 0x70, 0x07, enable
	};
	write_data(command, 5);
}

/**
 * Reads the status of the touch standard reference.
 * 
 * Returned Data:
 * 1. Data : 0x00 - 0x01 : 1 byte
 * 
 * @return none
 */
char GT1P_setTSRUsageStatusRead()
{
	unsigned char readData[1];
	unsigned char command[] = {
		0x1f, 0x4b, 0x70, 0x08
	};
	write_data(command, 4);
	
	read_USB(readData, 1);
	return readData[0];
}

/**
 * Store a touch setting package file to the module.
 * 
 * NOTE: The file to be stored must be a touch setting package file
 * provided by Noritake.
 * 
 * @param dest    The touch setting package destination.
 * @param TSPfile The touch setting package file to be stored.
 * @returns none
 */
void GT1P_storeTSP(char dest, unsigned char *TSPfile)
{
	int offset = 0;
	
	
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x1c, dest
	};
	write_data(command, 5);
	//write_data(TSPfile, 1024);
	
	while(offset < 1024)
	{
		write_data(TSPfile + offset, 64);
		offset += 64;
	}
}

/**
 * Select the desired touch setting package to be used.
 * The touch setting package must be stored in order
 * to be selected.
 * 
 * @param num The desired touch setting package number.
 * @returns none
 */
void GT1P_selectTSP(char num)
{
	unsigned char command[] = {
		0x1f, 0x4b, 0x70, 0x10, num
	};
	write_data(command, 5);
}

/**
 * Sets a memory switch to the passed in value.
 * 
 * memorySW = 0x05: Brightness level setting.
 * memorySW = 0x3B: Threshold level setting.
 * 
 * @param  memorySW The memory SW to be changed.
 * @param  data     The data to be written to the memory SW.
 * @return none
 */
void GT1P_memorySWSetting(char memorySW, char data){
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x03,
		memorySW, data
	};
	write_data(command, 6);
}

/**
 * Reads a memory switch value from the module.
 * 
 * @param  memorySW The memory switch to read.
 * @return char     The memory switch value.
 */
char GT1P_getMemorySWSetting(char memorySW)
{
	int count = 0;
	unsigned char readData[64];
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x04, memorySW
	};
	write_data(command, 5);
		
	while(4 != read_USB(readData, BULK_TRANSFER_MAX))
	{
		count++;
		if(count > 5)
		{
			printf("USB read error.\nPlease reconnect your module.\n");
			return 0xff;
		}
	}
	return readData[3];
}

/**
 * Initializes the display.
 * 
 * @return none
 */
void GT1P_Init(){
	write_char(0x1b);
	write_char(0x40);
}

/**
 * Enable or disable touch data transmission.
 * 
 * enable = 0x00: Transmit OFF
 * enable = 0x01: Transmit ON
 * Default: enable = 0x00
 * 
 * @param  enable Turn touch data ON or OFF.
 * @return none
 */
void GT1P_enableTouch(char enable){
	unsigned char command[] = {
		0x1f, 0x50, 0x20, enable
	};
	write_data(command, 4);
}

/**
 * Set the maximum number of simultaneous touch detection
 * 
 * numTouch = 0x00: Single-touch mode
 * numTouch = 0x01 - 0x0A: Multi-touch mode 
 * (numTouch = maximum number of touches)
 * 
 * @param  numTouch Number of touch points
 * @return none
 */
void GT1P_setTouchMode(char numTouch)
{
	unsigned char command[] = {
		0x1f, 0x50, 0x01, numTouch
	};
	write_data(command, 4);
}
