/*********************************************************
 * Filename: GT1P_USB_LIB.c
 * 
 * Purpose:
 * Provide wrapper functions to communicate with GT-1P
 * modules via libusb.h.
 * 
 * Version: 3.0 (ME-N63-01)
 * Date: 9/19/2018
 * Website: www.noritake-elec.com
 * (C) 2019 Noritake Co., Inc.
 *
 * Disclaimer: This software example is provided "AS-IS". We are not
 * responsible for any issues that may occur while using this program.
 * This software is intended for demonstration and evaluation purposes
 * only.
 *********************************************************/

#include <libusb-1.0/libusb.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "GT1P_USB_LIB.h"

#define VENDOR_ID 0x0eda
//Product ID definition not used in V1.1
#define PRODCT_ID 0x12de

static struct libusb_device_handle *devh = NULL;

//Bulk transfer used for everything besides touch data
static int ep_in_addr = 0x81; // 64 bytes max
static int ep_out_addr = 0x02; // 64 bytes max

//Product ID Global Variable
//Extern source found in GT1P_USB_LIB.h
int product_ID;

int USBactive = 0;

/**
 * Establish a USB connection with a GT-1P module.
 * 
 * TODO:
 * Try and make workaround for modprobe link from 
 * Noritake ISE's Qt application.
 * 
 * @return int USB connection success.
 */
int USB_start(){
	int rc;
	int result;
	int retVal;
	
	rc = libusb_init(NULL);
	if(rc < 0){
		fprintf(stderr, "Error initializing libusb: %s\n", libusb_error_name(rc));
		exit(1);
	}
	libusb_set_debug(NULL, 3);
	
	devh = libusb_open_device_with_vid_pid(NULL, VENDOR_ID, product_ID);
	if(!devh){
		//Try removing the usbserial module created on Noritake ISE demo kit.
		system("sudo modprobe -r usbserial");
		printf("Removed usbserial modprobe.\n");
		devh = libusb_open_device_with_vid_pid(NULL, VENDOR_ID, product_ID);
	}
	if(!devh){
		fprintf(stderr, "Error finding USB device.\n");
		fprintf(stderr, "Make sure your module is connected via USB.\n");
		if(devh){
			libusb_close(devh);
		}
		libusb_exit(NULL);
		retVal = 1;
	} else{
		result = libusb_claim_interface(devh, 1);
		if(result >= 0)
		{
			USBactive = 1;
			retVal = 0;
		} else {
			fprintf(stderr, "USB interface claim error %d\n", result);
			if(devh)
			{
				libusb_close(devh);
			}
			libusb_exit(NULL);
			retVal = 1;
		}
	}
	return retVal;
}

int USB_get_active()
{
	return USBactive;
}

/**
 * Sends a character over USB.
 * 
 * @param  c    The character to send.
 * @return none
 */
void write_char(unsigned char c){
	int actual_length;
	if(libusb_bulk_transfer(devh, ep_out_addr, &c, 1, &actual_length, 0) < 0){
		fprintf(stderr, "Error while sending char\n");
	}
}

/**
 * Sends multiple characters over USB.
 * 
 * @param  data  The byte data to send.
 * @return none
 */
void write_multi_char(unsigned char * data){
	int actual_length;
	if(libusb_bulk_transfer(devh, ep_out_addr, data, strlen((const char *)data), &actual_length, 0) < 0){
		fprintf(stderr, "Error while sending char\n");
	}
}

/**
 * Sends a character array of a determined size over USB.
 * This multiple send function is currently the most reliable.
 * 
 * @param  data The byte data array to send.
 * @param  size The size of the byte data array.
 * @return none
 */
void write_data(unsigned char * data, int size){
	int actual_length;
	if(libusb_bulk_transfer(devh, ep_out_addr, data, size, &actual_length, 0) < 0){
		fprintf(stderr, "Error while sending char\n");
	}
}

/**
 * Read data over USB.
 * 
 * @param  data The data array to store the read data.
 * @param  size The number of bytes to read.
 * @return Number of bytes received.
 */
int read_USB(unsigned char * data, int size){
	int actual_length;
	int result;
	result = libusb_bulk_transfer(devh, ep_in_addr, data, size, &actual_length, 1000);
	if(result < 0){
		
		switch(result)
		{
			case LIBUSB_ERROR_TIMEOUT:
				fprintf(stderr, "USB read timeout.\n");
				break;
			case LIBUSB_ERROR_PIPE:
				fprintf(stderr, "ERROR: USB endpoint has halted.\n");
				break;
			case LIBUSB_ERROR_OVERFLOW:
				fprintf(stderr, "ERROR: Device offered more data.\n");
				break;
			case LIBUSB_ERROR_NO_DEVICE:
				fprintf(stderr, "ERROR: No USB device found.\n");
				break;
			default:
				fprintf(stderr, "Error while reading data from USB.\n");
				break;
		}
	}
	return actual_length;
}

/**
 * Release the current USB connection.
 * 
 * @return none
 */
void release_USB(){
	libusb_release_interface(devh, 1);
	libusb_close(devh);
	libusb_exit(NULL);
	USBactive = 0;
}
