/*********************************************************
 * Purpose:
 * Provide functions directly relating to the GT-CP's
 * commands to be used via USB connection on a Linux
 * platform. 
 * 
 * Version: 1.0
 * Website: noritake-elec.com
 *********************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <libusb-1.0/libusb.h>
#include "GUTFT_USB_LIB.h"
#include "GUTFT.h"

/**
 * Initializes the display.
 * 
 * @return none
 */
void GUTFT_Init(){
	write_char(0x1b);
	write_char(0x40);
}

/**
 * Send a single byte to the display.
 * 
 * @param   c    The byte to send.
 * @returns none
 */
void GUTFT_putChar(char c){
	write_char(c);
}

/**
 * Send a string to the display.
 * 
 * @param   data The string to send.
 * @returns none
 */
void GUTFT_putString(unsigned char * data){
	write_multi_char(data);
}

/**
 * Turns the display OFF.
 * 
 * @return none
 */
void GUTFT_displayOFF(){
	unsigned char command[] = {
		0x1f, 0x28, 'a', 0x40, 0x00
	};
	write_data(command, 5);
}

/**
 * Turns the display ON.
 * 
 * @return none
 */
void GUTFT_displayON(){
	unsigned char command[] = {
		0x1f, 0x28, 'a', 0x40, 0x01
	};
	write_data(command, 5);
}

/**
 * Draw a line or box on the display.
 * 
 * mode = 0x00: Line
 * mode = 0x01: Box
 * mode = 0x02: Box fill
 * 
 * pen = 0x00: Pen color is set to the background color.
 * pen = 0x01: Pen color is set to the character color.
 * 
 * @param  mode   The desired drawing mode.
 * @param  pen    Set pen color to character or background color.
 * @param  xStart Line/Box drawing start position.
 * @param  yStart Line/Box drawing start position.
 * @param  xEnd   Line/Box drawing end position.
 * @param  yend   Line/Box drawing end position.
 * @return none 
 */
void GUTFT_drawLineBox(char mode, char pen, unsigned xStart, unsigned yStart, unsigned xEnd, unsigned yEnd){
	unsigned char command[] = {
		0x1f, 0x28, 0x64, 0x11,
		mode, pen, xStart, xStart >> 8,
		yStart, yStart >> 8, xEnd, xEnd >> 8,
		yEnd, yEnd >> 8
	};
	write_data(command, 14);
}

/**
 * Send an image to the display.
 * 
 * format = 0x81: Monochrome (1-bit) format
 * format = 0x86: Color 6-bit format
 * format = 0x8c: Color 12-bit format
 * format = 0x90: Color 16-bit format
 * format = 0x98: Color 24-bit format
 * format = 0xf0: BMP file format
 * 
 * @param  xSize     Bit image X size
 * @param  ySize     Bit image Y size
 * @param  format    The image format to be displayed.
 * @param  data      The image data to be displayed.
 * @param  imageSize The size of the image data being sent to the display.
 * @return none
 */
void GUTFT_drawImage(unsigned xSize, unsigned ySize, char format, unsigned char * data, unsigned imageSize){
	unsigned char drawCommand[] = {
		0x1f, 0x28, 0x66, 0x11, 
		xSize, (xSize >> 8), ySize, (ySize >> 8),
		format
	};
	printf("xSize in drawImage: %d\n", xSize);
	printf("ySize in drawImage: %d\n", ySize);
	printf("format in drawImage: %d\n", format);
	printf("imageSize in drawImage: %d\n", imageSize);
	write_data(drawCommand, 9);
	write_data(data, imageSize);
}

/**
 * Enable or disable touch data transmission.
 * 
 * enable = 0x00: Transmit OFF
 * enable = 0x01: Transmit ON
 * Default: enable = 0x00
 * 
 * @param  enable Turn touch data ON or OFF.
 * @return none
 */
void GUTFT_enableTouch(uint8_t enable){
	unsigned char command[] = {
		0x1f, 0x50, 0x20, enable
	};
	write_data(command, 4);
}

/**
 * Set the desired touch mode.
 * 
 * mode = 0x00: Single touch mode
 * 0x01 <= mode <= 0x0a: Multi-touch mode (up to 10 touches)
 * Default: mode = 0x00
 * 
 * @param  mode The desired touch mode.
 * @return none
 */
void GUTFT_multiTouchMode(char numTouch){
	unsigned char command[] = {
		0x1f, 0x50, 0x01, numTouch
	};
	write_data(command, 4);
}

/**
 * Set the corrdinate touch mode.
 * 
 * 0x00 <= channel <= 0x03
 * 
 * @param  channel The desired touch data channel.
 * @return none
 */
void GUTFT_coordinatesMode(uint8_t channel){
	unsigned char command[] = {
		0x1f, 0x50, 0x10, channel, 0x00
	};
	write_data(command, 5);
}

void GUTFT_customSwitchModeSingle(char channel, unsigned xPos, unsigned yPos, unsigned xSize, unsigned ySize)
{
	unsigned char command[] = {
		0x1f, 0x50, 0x10, channel, 0x02,
		0x01, xPos, xPos >> 8, yPos, yPos >> 8,
		xSize, xSize >> 8, ySize, ySize >> 8
	};
	write_data(command, 14);
}

/**
 * Performs a backspace on the GUTFT display.
 * 
 * @return none
 */
void GUTFT_back(){
	write_char(0x08);
}

/**
 * Moves the cursor forward one position on the GUTFT display.
 * 
 * @return none
 */
void GUTFT_forward(){
	write_char(0x09);
}

/**
 * Performs a line feed on the display.
 * 
 * @return none
 */
void GUTFT_lineFeed(){
	write_char(0x0a);
}

/**
 * Brings the cursor to its home position (top left) on the GUTFT display.
 * 
 * @return none 
 */
void GUTFT_home(){
	write_char(0x0b);
}

/**
 * Performs a carriage return on the GUTFT display.
 * 
 * @return none
 */
void GUTFT_carriageReturn(){
	write_char(0x0d);
}

/**
 * Performs a carriage return and line feed on the GUTFT display.
 * 
 * @return none
 */
void GUTFT_crlf(){
	unsigned char command[] = {
		0x0d, 0x0a
	};
	write_data(command, 2);
}

/**
 * Sends an x and y coordinate to the GUTFT display.
 * 
 * @param  x    The desired x coordinate.
 * @param  y    The desired y coordinate.
 * @return none
 */
void GUTFT_XY(unsigned x, unsigned y){
	unsigned char command[] = {
		x, x >> 8, y, y >> 8
	};
	write_data(command, 4);
}

/**
 * Moves the cursor to a specific location on the GUTFT module.
 * 
 * @param  x    The desired x coordinate.
 * @param  y    The desired y coordinate.
 * @return none
 */
void GUTFT_setCursor(unsigned x, unsigned y){
	unsigned char command[] = {
		0x1f, '$'
	};
	write_data(command, 2);
	GUTFT_XY(x, y);
}

/**
 * Clears the screen of the GUTFT module.
 * 
 * @return none
 */
void GUTFT_clearScreen(){
	write_char(0x0c);
}

/**
 * Clear the current line. Cursor will move to the left end of the current line.
 * 
 * @return none
 */
void GUTFT_lineClear(){
	write_char(0x18);
}

/**
 * Clear the current line. Cursor will move to the right end of the current line.
 * 
 * @return none
 */
void GUTFT_lineEndClear(){
	write_char(0x19);
}

/**
 * Turns the cursor ON.
 * 
 * @return none
 */
void GUTFT_cursorON(){
	unsigned char command[] = {
		0x1f, 'C', 0x01
	};
	write_data(command, 3);
}

/**
 * Turns the cursor OFF.
 * 
 * @return none
 */
void GUTFT_cursorOFF(){
	unsigned char command[] = {
		0x1f, 'C', 0x00
	};
	write_data(command, 3);
}

/**
 * Select how display actions will affect the module.
 * 
 * Display screen mode: Display action is valid within the Display or hidden area.
 * All screen mode: Display action is valid over the entire display memory.
 * 
 * mode = 0x00: Display screen mode
 * mode = 0x01: All screen mode
 * Default: mode = 0x00 or memory SW setting
 * 
 * @param  mode The desired write screen mode.
 * @return none
 */
void GUTFT_writeScreenMode(char mode){
	unsigned char command[] = {
		0x1f, 0x28, 0x77, 0x10, mode
	};
	write_data(command, 5);
}

/**
 * Select how backgound pixels are written to the display memory.
 * 
 * mode = 0x10: Normal display write (background pixels are written to display memory)
 * mode = 0x11; Thru write (background pixels are not written to display memory)
 * Default: mode = 0x10
 * 
 * @param  mode The desired mixture display mode.
 * @return none
 */
void GUTFT_writeMixtureDisplayMode(char mode){
	unsigned char command[] = {
		0x1f, 0x77, mode
	};
	write_data(command, 3);
}

/**
 * Sets the multiple byte character set to be used.
 * 
 * Code for each character set:
 * 0x00: Japanese
 * 0x01: Korean
 * 0x02: Simplified Chinese
 * 0x03: Traditional Chinese
 * 
 * @param  code The character set code to be used.
 * @return none
 */
void GUTFT_useMultiByteChars(char enable){
	unsigned char command[] = {
		0x1f, 0x28, 'g', 0x20, enable
	};
	write_data(command, 5);
}

/**
 * Sets the multiple byte character set to be used.
 * 
 * Code for each character set:
 * 
 * 0x00 - Japanese
 * 0x01 - Korean
 * 0x02 - Simplified Chinese
 * 0x03 - Traditional Chinese
 * 
 * @param  code  The character set code to be used.
 * @return none
 */
void GUTFT_setMultiByteCharSet(char code){
	unsigned char command[] = {
		0x1f, 0x28, 'g', 0x0f, code
	};
	write_data(command, 5);
}

/**
 * Enables or disables the use of custom characters.
 * 
 * @param  enable Enables or disable custom characters.
 * @return none
 */
void GUTFT_useCustomChars(char enable){
	unsigned char command[] = {
		0x1b, '%', enable
	};
	write_data(command, 3);
}

/**
 * Stores a custom character on the display.
 * 
 * To print a custom character, write the code of the character
 * to the display after using this function.
 * 
 * code starts at 0x20 and ends at 0x27
 * format = 0x01: 6x8 pixel character
 * format = 0x02: 8x16 pixel character
 * format = 0x03: 12x24 pixel character
 * format = 0x04: 16x32 pixel character
 * 
 * Data format for custom characters:
 * 
 * 6x8 character
 * 0b00000000
 * 0b00000000
 * 0b00000000
 * 0b00000000
 * 0b00000000
 * 0b00000000
 * 
 * 8x16 character
 * 0b00000000, 0b00000000
 * 0b00000000, 0b00000000
 * 0b00000000, 0b00000000
 * 0b00000000, 0b00000000
 * 0b00000000, 0b00000000
 * 0b00000000, 0b00000000
 * 0b00000000, 0b00000000
 * 0b00000000, 0b00000000
 * 
 * 12x24 character
 * 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000
 * 
 * 16x32 character
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 0b00000000, 0b00000000, 0b00000000, 0b00000000
 * 
 * LSB is the bottom of the character.
 * MSP is the top of the character.
 * First byte of data is the left most column of the character
 * Last byte of data is the right most column of the character.
 * 
 * @param  codeStart The starting code value for the character being defined.
 * @param  codeEnd   The ending code value for the character being defined.
 * @param  format    The format of the character being defined.
 * @param  xDir      The number of pixels for the x-direction of the character.
 * @param  data      Data of the character being defined.
 * @return none
 */
void GUTFT_defineCustomChar(char codeStart, char codeEnd, char format, char xDir, unsigned char * data){
	unsigned char command[] = {
		0x1b, '&', format, codeStart,
		codeEnd, xDir
	};
	write_data(command, 6);
	switch(format){
		case 0x01: //6x8 Pixel Character
			write_data(data, xDir);
			break;
		case 0x02: //8x16 Pixel Character
			write_data(data, xDir * 2);
			break;
		case 0x03: //12x24 Pixel Character
			write_data(data, xDir * 3);
			break;
		case 0x04: //16x32 Pixel Character
			write_data(data, xDir * 4);
			break;
	}
}

/**
* Deletes a previously defined custom character.
* 
* type = 0x01: Select 6x8 pixel character.
* type = 0x02: Select 8x16 pixel character.
* type = 0x03: Select 12x24 pixel character.
* type = 0x04: Select 16x32 pixel character.
* 0x20 <= code <= 0xff
* 
* @param  type    The custom character type to be deleted.
* @param  code    Address of the custom character to be deleted.
* @return none
*/
void GUTFT_deleteCustomChar(char type, char code){
	unsigned char command[] = {
		0x1b, '?', type, code
	};
	write_data(command, 4);
}

/**
* Sets the desired ASCII variant to be used.
* Also called "International font select" in the GUTFT software manual.
* Font codes:
* 
* 0x00 - America
* 0x01 - France
* 0x02 - Germany
* 0x03 - England
* 0x04 - Denmark 1
* 0x05 - Sweden
* 0x06 - Italy
* 0x07 - Spain 1
* 0x08 - Japan
* 0x09 - Norway
* 0x0A - Denmark 2
* 0x0B - Spain 2
* 0x0C - Latin America
* 0x0D - Korea
* 
* 
* @param  code   The code of the ASCII variant to be used.
* @return none
*/
void GUTFT_setAsciiVariant(char code){
	unsigned char command[] = {
		0x1b, 'R', code
	};
	if(code < 0x0d){
		write_data(command, 3);
	}
}

/**
* Sets the desired character set.
* Character set codes:
* 
* 0x00 - PC437(USA-Euro std)
* 0x01 - Katakana - Japanese
* 0x02 - PC850 (Multilingual)
* 0x03 - PC860 (Portuguese)
* 0x04 - PC863 (Canadian-French)
* 0x05 - PC865 (Nordic)
* 0x10 - WPC1252 (Latin)
* 0x11 - PC866 (Cryllic #2)
* 0x12 - PC852 (Latin 2)
* 0x13 - PC858 (Eastern European)
* 0xFE - UTF-8 input
* 0xFF - User table
* 
* 
* @param  code   The code for the desired character set.
* @return none
*/
void GUTFT_setCharSet(char code){
	unsigned char command[] = {
		0x1b, 't', code
	};
	if(code < 0x05 || (0x10 <= code && code <= 0x13) || code == 0xfe || code == 0xff){
		write_data(command, 3);
	}
}

/**
* Sets the scroll mode on the GUTFT module.
* Scroll modes:
* <ul>
* <li> 0x01 - Wrapping mode (overwrite mode)
* <li> 0x02 - Vertical scroll mode
* <li> 0x03 - Horizontal scroll mode
* <li> 0x05 - Horizontal scroll mode, scroll ON)
* </ul>
* 
* @param  mode   The desired scroll mode.
* @return none
*/
void GUTFT_setScrollMode(char mode){
	unsigned char command[] = {
		0x1f, mode
	};
	write_data(command, 2);
}

/**
* Sets the horizontal scroll speed on the GUTFT series module.
* Speed parameter specifics:
* <ul>
* <li> 0x00 <= speed <= 0x1f
* <li> speed = 0x00 : Instantaneous Speed
* <li> speed = 0x01 : IntTime / 2 pixels
* <li> speed = 0x02 - 0x1f : (n-1) * T ms / pixel
* </ul>
* 
* @param  speed  The desired horizontal scroll speed.
* @return none
*/
void GUTFT_setHorizScrollSpeed(char speed){
	unsigned char command[] = {
		0x1f, 's', speed
	};
	write_data(command, 3);
}

/**
* Turns the display inversion OFF.
* 
* @return none
*/
void GUTFT_invertOFF(){
	unsigned char command[] = {
		0x1f, 'r', 0x00
	};
	write_data(command, 3);
}

/**
* Turns the display inversion ON.
* 
* @return none
*/
void GUTFT_invertON(){
	unsigned char command[] = {
		0x1f, 'r', 0x01
	};
	write_data(command, 3);
}

/**
* Sets the composition mode of the GUTFT module.
* 
* Composition modes:
* <ul>
* <li> 0x00 - Normal display write (not mixture display)
* <li> 0x01 - OR display write
* <li> 0x02 - AND display write
* <li> 0x03 - XOR display write
* </ul>
* 
* @param  mode  The desired composition mode.
* @return none
*/
void GUTFT_setCompositionMode(char mode){
	unsigned char command[] = {
		0x1f, 'w', mode
	};
	write_data(command, 3);
}

/**
* Sets the screen brightness of the GUTFT module.
* 
* Screen brightness levels:
* <ul>
* <li> 0x01 - 12.5%
* <li> 0x02 - 25.0%
* <li> 0x03 - 37.5%
* <li> 0x04 - 50.0%
* <li> 0x05 - 62.5%
* <li> 0x06 - 75.0%
* <li> 0x07 - 87.5%
* <li> 0x08 - 100%
* </ul>
* 
* @param  level The desired brightness level.
* @return none
*/
void GUTFT_setScreenBrightness(unsigned level){
	if(level == 0){
		GUTFT_displayOFF();
	}else if(level <= 100){
		unsigned char command[] = {
			0x1f, 'X', ((level * 10 + 120) / 125)
		};
		GUTFT_displayON();
		write_data(command, 3);
	}
}

/**
* Suspends the program and waits for a user defined amount of time.
* Wait time = t * approximately 0.47s
* 
* @param  wait    The amount of time for the MCU to wait.
* @return none
*/
void GUTFT_wait(char wait){
	unsigned char command[] = {
		0x1f, 0x28, 'a', 0x01, wait
	};
	write_data(command, 5);
}

/**
* Suspends the program and waits for a user defined amount of time.
* Wait time = t * IntTime
* 
* @param  time The amount of time to wait.
* @return none
*/
void GUTFT_shortWait(char time){
	unsigned char command[] = {
		0x1f, 0x28, 'a', 0x02, time
	};
	write_data(command, 5);
}

/**
* Sets the scroll parameters and scrolls the screen.
* 
* @param  x       The amount of pixels for the screen to scroll horizontally.
* @param  y       The amount of pixels for the screen to scroll vertically.
* @param times    The number of times the screen will scroll?
* @param speed    The speed at which scrolling occurs.
* @return none
*/
void GUTFT_scrollScreen(unsigned x, unsigned y, unsigned times, char speed){
	unsigned char command1[] = {
		0x1f, 0x28, 'a', 0xa0
	};
	unsigned char command2[] = {
		times, times >> 8, speed
	};
	write_data(command1, 4);
	GUTFT_XY(x, y);
	write_data(command2, 3);
}

/**
* Start a curtain display action on-screen.
* <ul>
* <li> direction = 0x00: To the right from the left edge.
* <li> direction = 0x01: To the left from the right edge.
* <li> direction = 0x02: To the left and right separately from the center.
* <li> direction = 0x03: To the center from the left and right edge.
* </ul>
* 
* <ul>
* <li> Curtain action speed = 60 * speed * IntTime
* <li> 0x00 <= speed <= 0xff
* </ul>
* 
* <ul>
* <li> 0x00 <= red <= 0xff
* <li> 0x00 <= green <= 0xff
* <li> 0x00 <= blue <= 0xff
* </ul>
* 
* @param  direction Direction of curtain action.
* @param  speed     Curtain action speed.
* @param  red       Red brightness
* @param  green     Green brightness
* @param  blue      Blue brightness
* @return none
*/
void GUTFT_curtainAction(char direction, char speed, char red, char green, char blue){
	unsigned char command[] = {
		0x1f, 0x28, 0x61, 0xa2,
		direction, speed, red, green, blue
	};
	write_data(command, 9);
}

/**
* Start a curtain display action on-screen.
* <ul>
* <li> direction = 0x00: To the right from the left edge.
* <li> direction = 0x01: To the left from the right edge.
* <li> direction = 0x02: To the left and right separately from the center.
* <li> direction = 0x03: To the center from the left and right edge.
* </ul>
* 
* <ul>
* <li> Curtain action speed = 60 * speed * IntTime
* <li> 0x00 <= speed <= 0xff
* </ul>
* 
* @param  direction Direction of curtain action.
* @param  speed     Spring action speed.
* @param  x         Display memory X position.
* @param  y         Display memory y position.
* @return none
*/
void GUTFT_springAction(char direction, char speed, unsigned x, unsigned y){
	unsigned char command[] = {
		0x1f, 0x28, 0x61, 0xa3,
		direction, speed
	};
	write_data(command, 6);
	GUTFT_XY(x, y);
}

/**
* Start a checkerboard display action.
* <ul>
* <li> 0x00 <= actionType <= 0x02
* <li> actionType = 0x00: Transition pattern 1
* <li> actionType = 0x01: Transition pattern 2
* <li> actionType = 0x02: Transition pattern 3
* <li> 0x00 <= speed <= 0xff
* </ul>
* 
* @param  actionType The desired transition pattern.
* @param  speed      Checkerboard action speed.
* @param  x          Display memory X position.
* @param  y          Display memory y position.
* @return none
*/
void GUTFT_randomAction(char actionType, char speed, unsigned x, unsigned y){
	unsigned char command[] = {
		0x1f, 0x28, 0x61, 0xa4,
		actionType, speed
	};
	write_data(command, 6);
	GUTFT_XY(x, y);
}

/**
* Start a fade-in display action.
* <ul>
* <li> Fade-in time = speed * approximately 0.5s
* </ul>
* 
* @param  speed Fade-in action speed.
* @param  x     Display memory X position.
* @param  y     Display memory Y position.
* @return none
*/
void GUTFT_fadeInAction(char speed, unsigned x, unsigned y){
	unsigned char command[] = {
		0x1f, 0x28, 0x61, 0xa5,
		speed
	};
	write_data(command, 5);
	GUTFT_XY(x, y);
}

/**
 * Start a fade-out display action.
 * 
 * Fade-out time = speed * approximately 0.5s
 * 
 * @param  speed Fade-out action speed
 * @return none
 */
void GUTFT_fadeOutAction(char speed){
	unsigned char command[] = {
		0x1f, 0x28, 0x61, 0xa6,
		speed
	};
	write_data(command, 5);
}

/**
 * Resets the blink settings on the module to all zeros.
 * 
 * @return none
 */
void GUTFT_blinkScreenOFF(){
	unsigned char command[] = {
		0x1f, 0x28, 'a', 0x11,
		0x00, 0x00, 0x00, 0x00
	};
	write_data(command, 8);
}

/**
 * Blinks the GUTFT series module screen based on user input parameters.
 * 
 * 0x01 <= onTime <= 0xff
 * 0x01 <= offTime <= 0xff
 * 0x00 <= cycles <= 0xff
 * 
 * @param  enable  Enables or disables screen blinking.
 * 	0x00 - Normal display
 * 	0x01 - Blink display (alternatively Normal and Blank display)
 * 	0x02 - Reserved
 * @param  reverse Enables of disables the reverse screen pattern.
 * @param  onTime  The time that the display stays ON during blinking.
 * @param  offTime The time that the display stays OFF during blinking.
 * @return none
 */
void GUTFT_blinkScreenON(char enable, char reverse, char onTime, char offTime, char cycles){
	unsigned char command[] = {
		0x1f, 0x28, 'a', 0x11,
		(enable? (reverse? 2: 1): 0), onTime, offTime, cycles
	};
	write_data(command, 8);
}

/**
 * Sets the screen saver mode for the GUTFT series module.
 * 
 * Screen saver modes:
 * 0x00 - Display power OFF
 * 0x01 - Display power ON
 * 0x02 - All dots OFF
 * 0x03 - All dots ON
 * 0x04 - Repeat blink display with normal and reverse display.
 * 
 * @param  mode The desired screensaver mode. (see ScreenSaver)
 * @return none
 */
void GUTFT_screenSaver(char mode){
	unsigned char command[] = {
		0x1f, 0x28, 'a', 0x40, mode
	};
	write_data(command, 5);
}

/**
 * Sets the font stype for the GUTFT series module.
 * 
 * @param  proportional Enable or disable a proportional font style.
 * @param  evenSpacing  Enable or disable even spacing between characters.
 * @return none
 */
void GUTFT_setFontStyle(char proportional, char evenSpacing){
	unsigned char command[] = {
		0x1f, 0x28, 'g', 0x03, (proportional * 2 + evenSpacing)
	};
	write_data(command, 5);
}

/**
 * Sets the desired 1-byte character font size
 */
void GUTFT_setFontSize(char size){
	unsigned char command[] = {
		0x1f, 0x28, 0x67, 0x01, size
	};
	write_data(command, 5);
}

/**
 * Set the desired outline font size.
 * 
 * 0x0001 <= charLineHeight <= Ydots
 * 0x0001 <= nominalCharY <= 0x03ff
 * 0x0001 <= nomunalCharX <= 0x03ff
 * 0x0001 <= baseOffsetY <= 0x03ff | if 0x8000: zero offset
 * 
 * @param  charLineHeight Y-size (height) of character line (line spacing), in pixels
 * @param  nominalCharY   Nominal character Y-size (height), in pixels.
 * @param  nominalCharX   Nominal character X-size (width), in pixels.
 * @param  baseOffsetY    Baseline Y-offset from cursor position, in pixels.
 * @return none
 */
void GUTFT_setOutlineFontSize(unsigned charLineHeight, unsigned nominalCharY, unsigned nominalCharX, unsigned baseOffsetY){
	unsigned char command[] = {
		0x1f, 0x28, 0x67, 0x06, charLineHeight, charLineHeight >> 8,
		nominalCharY, nominalCharY >> 8, nominalCharX, nominalCharX >> 8,
		baseOffsetY, baseOffsetY >> 8
	};
	write_data(command, 12);
	//GUTFT_XY(charLineHeight, nominalCharY);
	//GUTFT_XY(nominalCharX, baseOffsetY);
}

/**
 * Set the outline font type. 
 * This needs to be used in order to use outline fonts.
 * 
 * fontType = 0x00: Japanese
 * fontType = 0x01: Korean
 * fontType = 0x02: Simplified Chinese
 * fontType = 0x03: Traditional Chinese
 * fontType = 0x80: None (no outline font selected)
 * fontType = 0xff: User-supplied font file
 * Default fontType = 0x80
 * 
 * @param  fontType The desired outline font type.
 * @return none
 */
void GUTFT_setOutlineFontType(char fontType){
	unsigned char command[] = {
		0x1f, 0x28, 0x67, 0x08, fontType
	};
	write_data(command, 5);
}

/**
 * Sets the font size for the GUTFT series module.
 * 
 * @param  x    X magnification factor
 * @param  y    Y magnification factor
 * @return none
 */
void GUTFT_setFontMagnification(char x, char y){
	unsigned char command[] = {
		0x1f, 0x28, 'g', 0x40, x, y
	};
	if(x <= 4 && y <= 4){
		write_data(command, 6);
	}
}

/**
 * Set the character style.
 * 
 * style = 0x00: Normal
 * style = 0x01: Bold
 * style = 0x02: Shadow
 * style = 0x03: Bordering
 * Default: style = 0x00
 * 
 * @param  style  The desired character style
 * @return none
 */
void GUTFT_setCharacterStyle(char style){
	unsigned char command[] = {
		0x1f, 0x28, 0x67, 0x41, style
	};
	write_data(command, 5);
}

/**
 * Sets the character color.
 * 
 * 0x00 = No brightness
 * 0x7f = 50% brightness
 * 0xff = Maximum brightness
 * 
 * @param  red   Red brightness
 * @param  green Green brightness
 * @param  blue  Blue brightness
 * @return none
 */
void GUTFT_setCharacterColor(char red, char green, char blue){
	unsigned char command[] = {
		0x1f, 0x28, 0x67, 0x50, red, green, blue
	};
	write_data(command, 7);
}

/**
 * 
 */
void GUTFT_backgroundColorEnable(char enable)
{
	unsigned char command[] = {
		0x1f, 0x28, 0x67, 0x58, enable
	};
	write_data(command, 5);
}

/**
 * Set the character background (highlight) color.
 * 
 * 0x00 = No brightness
 * 0x7f = 50% brightness
 * 0xff = Maximum brightness
 * 
 * @param  red   Red brightness
 * @param  green Green brightness
 * @param  blue  Blue brightness
 * @return none 
 */
void GUTFT_setBackgroundColor(char red, char green, char blue){
	unsigned char command[] = {
		0x1f, 0x28, 0x67, 0x51, red, green, blue
	};
	write_data(command, 7);
}

/**
 * Set the character shadow and bordering color.
 * 
 * 0x00 = No brightness
 * 0x7f = 50% brightness
 * 0xff = Maximum brightness
 * 
 * @param  red   Red brightness
 * @param  green Green brightness
 * @param  blue  Blue brightness
 * @return none
 */
void GUTFT_setShadowBorderingColor(char red, char green, char blue){
	unsigned char command[] = {
		0x1f, 0x28, 0x67, 0x52, red, green, blue
	};
	write_data(command, 7);
}

/**
 * Turns the background ON or OFF.
 * 
 * When the background is OFF, its color is always black.
 * When the background is ON, its color is determined by GUTFT_setBackgroundColor.
 * enable = 0x00: Background OFF
 * enable = 0x01: Background ON
 * 
 * @param  enable Enable of disable a colored background.
 * @return none
 */
void GUTFT_setBackground(char enable){
	unsigned char command[] = {
		0x1f, 0x28, 0x67, 0x58, enable
	};
	write_data(command, 5);
}

/**
 * Selects the current window.
 * 
 * @param  window The desired window to select. (0-4)
 * @return none
 */
void GUTFT_selectWindow(char window){
	unsigned char command[] = {
		0x1f, 0x28, 0x77, 0x01, window
	};
	if(window <= 4){
		write_data(command, 5);
	}
}

/**
 * Defines a specific window for the GUTFT series module.
 * 
 * @param  window The number of the window to be created. (1-4)
 * @param  x      The x coordinate of the new window.
 * @param  y      The y coordinate of the new window.
 * @param  width  The width of the new window.
 * @param  height The height of the new window.
 * @return none
 */
void GUTFT_defineWindow(char window, unsigned x, unsigned y, unsigned width, unsigned height){
	unsigned char command[] = {
		0x1f, 0x28, 'w', 0x02, window, 0x01
	};
	write_data(command, 6);
	GUTFT_XY(x, y);
	GUTFT_XY(width, height);
}

/**
 * Deletes a previously defined window.
 * 
 * @param  window The number of the window to be deleted. (1-4)
 * @return none
 */
void GUTFT_deleteWindow(char window){
	unsigned char command[] = {
		0x1f, 0x28, 'w', 0x02, window, 0x00
	};
	write_data(command, 6);
	GUTFT_XY(0, 0);
	GUTFT_XY(0, 0);
}

/**
 * Joins all of the screens into one base screen.
 * 
 * @return none
 */
void GUTFT_joinScreens(){
	unsigned char command[] = {
		0x1f, 0x28, 'w', 0x10, 0x01
	};
	write_data(command, 5);
}

/**
 * Separates all of the screens into four separate screens.
 * 
 * @return none
 */
void GUTFT_separateScreens(){
	unsigned char command[] = {
		0x1f, 0x28, 'w', 0x10, 0x00
	};
	write_data(command, 5);
}

/**
 * Software datasheet function: Downloaded bit image display
 * Display an image stored in system memory.
 * 
 * memory = 0x00: RAM bit image
 * memory = 0x01: FROM1 bit image
 * memory = 0x02: Display memory
 * memory = 0x10: FROM2 base address 0x0000.0000
 * memory = 0x11: FROM2 base address 0x0100.0000
 * memory = 0x12: FROM2 base address 0x0200.0000
 * memory = 0x13: FROM2 base address 0x0300.0000
 * memory = 0x14: FROM2 base address 0x0400.0000
 * memory = 0x15: FROM2 base address 0x0500.0000
 * memory = 0x16: FROM2 base address 0x0600.0000
 * memory = 0x17: FROM2 base address 0x0700.0000
 * memory = 0x18: FROM2 base address 0x0800.0000
 * memory = 0x19: FROM2 base address 0x0900.0000
 * memory = 0x1a: FROM2 base address 0x0a00.0000
 * memory = 0x1b: FROM2 base address 0x0b00.0000
 * memory = 0x1c: FROM2 base address 0x0c00.0000
 * memory = 0x1d: FROM2 base address 0x0d00.0000
 * memory = 0x1e: FROM2 base address 0x0e00.0000
 * memory = 0x1f: FROM2 base address 0x0f00.0000
 * 
 * format = 0x81: Monochrome (1-bit) format
 * format = 0x86: Color 6-bit format
 * format = 0x8c: Color 12-bit format
 * format = 0x90: Color 16-bit format
 * format = 0x91: Color 16-bit high-speed format
 * format = 0x98: Color 24-bit format
 * format = 0xf0: BMP file format
 * 
 * @param  memory    The type of memory where the image is stored.
 * @param  address   The memory address of the image.
 * @param  extension The address extension.
 * @param  xDefine   The defined width to be displayed of the image.
 * @param  xSize     The width of the image.
 * @param  ySize     The height of the image.
 * @param  format    The desired image format.
 * @return none
 */
void GUTFT_storedBitImageDraw(char memory, unsigned  address, unsigned extension, unsigned xDefine, unsigned xSize, unsigned ySize, char format){
	unsigned char command[] = {
		0x1f, 0x28, 0x66, 0x10,
		memory, address, address >> 8, extension,
		xDefine, xDefine >> 8, xSize, xSize >> 8,
		ySize, ySize >> 8, format
	};
	write_data(command, 15);
}

/**
 * Draw a pixel on the display.
 * 
 * pen = 0x00: Pen color is set to the background color.
 * pen = 0x01: Pen color is set to the character color.
 * 
 * @param  pen Set pen color to character or background color.
 * @param  x   Pixel position X.
 * @param  y   Pixel position Y.
 * @return none
 */

void GUTFT_pixelDrawing(char pen, unsigned x, unsigned y){
	unsigned char command[] = {
		0x1f, 0x28, 0x64, 0x10, pen
	};
	write_data(command, 5);
	GUTFT_XY(x, y);
}

/**
 * Set the switches matrix mode.
 * 
 * 0x00 <= channel <= 0x03
 * 0x01 <= switchesX <= 0x10
 * 0x01 <= switchesY <= 0x10
 * 0x01 <= clearanceX <= 0x10
 * 0x01 <= clearanceY <= 0x10
 * 
 * @param  channel    The desired touch data channel.
 * @param  switchesX  The number of switches on the x-axis.
 * @param  switchesY  The number of switches on the y-axis.
 * @param  clearanceX The size of the non-responsive area in-between switches on the x-axis.
 * @param  clearanceY The size of the non-responsive area in-between switches on the y-axis.
 * @return none
 */
void GUTFT_switchMatrixMode(char channel, char switchesX, char switchesY, char clearanceX, char clearanceY){
	unsigned char command[] = {
		0x1f, 0x50, 0x10, channel,
		0x01, switchesX, clearanceX, switchesY,
		clearanceY
	};
	write_data(command, 9);
}

/**
 * Function to set the GPIO ports on the GUTFT unit to be inputs or outputs.
 * 
 * Settings:
 * portNumber = 0x00: Port 0 (GPIO 0-7)
 * portNumber = 0x01: Port 1 (GPIO 8-15)
 * portNumber = 0x02: Port 2 (GPIO 16-32)
 * portNumber = 0x03: POrt 3 (GPIO 25, 25)
 * portSetting = 0x00 (All input) to 0x0f (all output)
 * 0x00 <= portSetting <= 0xff
 * The portSetting parameter is bit-wise to the ports present on the module.
 * 
 * @param  portNumber  The desired port number to set.
 * @param  portSetting The passed in port setting.
 * @return none
 */
void GUTFT_IOPortSetting(char portNumber, char portSetting){
	unsigned char command[] = {
		0x1f, 0x28, 0x70, 0x01,
		portNumber, portSetting
	};
	write_data(command, 6);
}

/**
 * Function to set the output value on the GPIO ports.
 * 
 * Settings:
 * portNumber = 0x00: Port 0 (GPIO 0-7)
 * portNumber = 0x01: Port 1 (GPIO 8-15)
 * portNumber = 0x02: Port 2 (GPIO 16-32)
 * portNumber = 0x03: Port 3 (GPIO 24, 25)
 * 0x00 <= portValue <= 0xff
 * 
 * @param  portNumber The desired port number to set.
 * @param  portValue  The value to put on the port.
 * @return none
 */
void GUTFT_IOPortOutput(char portNumber, char portValue){
	unsigned char command[] = {
		0x1f, 0x28, 0x70, 0x10,
		portNumber, portValue
	};
	write_data(command, 6);
}

/**
 * Function to request the GUTFT module to read the values present on the GPIO ports.
 * 
 * portNumber = 0x00: Port 0 (GPIO 0-7)
 * portNumber = 0x01: Port 1 (GPIO 8-15)
 * portNumber = 0x02: Port 2 (GPIO 16-32)
 * portNumber = 0x03: Port 3 (GPIO 24, 25)
 * 
 * @param  portNumber The port number to read.
 * @return none
 */
void GUTFT_IOPortInput(char portNumber){
	unsigned char command[] = {
		0x1f, 0x28, 0x70, 0x20,
		portNumber
	};
	write_data(command, 5);
}

/**
 * Sets a memory switch to the passed in value.
 * 
 * @param  memorySW The memory SW to be changed.
 * @param  data     The data to be written to the memory SW.
 * @return none
 */
void GUTFT_memorySWSetting(char memorySW, char data){
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x03,
		memorySW, data
	};
	write_data(command, 6);
}

/**
 * Set multiple memory swtich values.
 * 
 * 0x01 <= numSettings <= 0xff
 * data should be set up like:
 * memory SW number [1], setting value [1],
 * memory SW number [2], setting value [2],
 * ... memory SW number [n], setting value [n]
 * 
 * @param  numSettings The number of SW settings to be set.
 * @param  data        The setting data.
 * @return none
 */
void GUTFT_memorySWSettingMulti(char numSettings, unsigned char * data){
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x03, 0xff, numSettings
	};
	write_data(command, 6);
	write_data(data, numSettings * 2);
}

/**
 * Send the contents of memory SW data.
 * 
 * The following data is transmitted:
 * 1. Header       | 0x28      | 1 byte
 * 2. Identifier 1 | 0x65      | 1 byte
 * 3. Identifier 2 | 0x04      | 1 byte
 * 4. Data         | 0x00-0xff | 1 byte / n bytes
 * 
 * @param  switchNum The memory SW to be read.
 * @return none
 */
void GUTFT_memorySWDataSend(char switchNum){
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x04,
		switchNum
	};
	write_data(command, 5);
}

/**
 * Send the contents of memory SW data.
 * 
 * 0x01 <= numReads <= 0xff
 * 
 * @param  numReads The number of reads to perform.
 * @param  data     The memory SW numbers to be read.
 * @return none
 */
void GUTFT_memorySWDataSendMulti(char numReads, unsigned char * data){
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x04,
		0xff, numReads
	};
	write_data(command, 6);
	write_data(data, numReads);
}

/**
 * Defines a 16x16 pixel downloaded character (2-byte character) in the code
 * specified by codeUpper and codeLower.
 * 
 * codeUpper, codeLower: Depends on currently selected language.
 * 
 * codeUpper = 0xec && 0x40 <= codeLower <= 0x4f : Japanese - JIS X0208 (SHIFT-JIS)
 * codeUpper = 0xfe && 0xa1 <= codeLower <= 0xb0 : Korean - KSC5601-87
 * codeUpper = 0xfe && 0xa1 <= codeLower <= 0xb0 : Simplified Chinese - GB2312-80
 * codeUpper = 0xfe && 0xa1 <= codeLower <= 0xb0 : Traditional Chinese - Big-5
 * 
 * @param  codeUpper Character code, upper byte
 * @param  codeLower Character code, lower byte
 * @param  data      Character definition data
 * @return none
 */
void GUTFT_16x16CharacterDefinition(char codeUpper, char codeLower, unsigned char * data){
	unsigned char command[] = {
		0x1f, 0x28, 0x67, 0x10,
		codeUpper, codeLower
	};
	write_data(command, 6);
	write_data(data, 32);
}

/**
 * Deletes a defined 16x16 download character in code specified by codeUpper and codeLower.
 * 
 * codeUpper, codeLower: Depends on currently selected language.
 * 
 * codeUpper = 0xec && 0x40 <= codeLower <= 0x4f : Japanese - JIS X0208 (SHIFT-JIS)
 * codeUpper = 0xfe && 0xa1 <= codeLower <= 0xb0 : Korean - KSC5601-87
 * codeUpper = 0xfe && 0xa1 <= codeLower <= 0xb0 : Simplified Chinese - GB2312-80
 * codeUpper = 0xfe && 0xa1 <= codeLower <= 0xb0 : Traditional Chinese - Big-5
 * 
 * @param  codeUpper Character code, upper byte
 * @param  codeLower Character code, lower byte
 * @return none
 */
void GUTFT_16x16CharacterDelete(char codeUpper, char codeLower){
	unsigned char command[] = {
		0x1f, 0x28, 0x67, 0x11,
		codeUpper, codeLower
	};
	write_data(command, 6);
}

/**
 * Defines a 32x32 pixel downloaded character (2-byte character) in character code specified
 * by codeUpper and codeLower.
 * 
 * codeUpper, codeLower: Depends on currently selected language.
 * 
 * codeUpper = 0xec && 0x40 <= codeLower <= 0x4f : Japanese - JIS X0208 (SHIFT_JIS)
 * 
 * @param  codeUpper Character code, upper byte
 * @param  codeLower Character code, lower byte
 * @param  data      Character definition data
 * @return none
 */
void GUTFT_32x32CharacterDefinition(char codeUpper, char codeLower, unsigned char * data){
	unsigned char command[] = {
		0x1f, 0x28, 0x67, 0x14,
		codeUpper, codeLower
	};
	write_data(command, 6);
	write_data(data, 128);
}

/**
 * Delete a defined 32x32 download character in code specified by codeUpper and codeLower.
 * This command is invalid if Japanese is not the seleted language.
 * 
 * codeUpper, codeLower: Depends on currently selected language.
 * 
 * codeUpper = 0xec && 0x40 <= codeLower <= 0x4f : Japanese - JIS X0208 (SHIFT_JIS)
 * 
 * @param  codeUpper Character code, upper byte
 * @param  codeLower Character code, lower byte
 * @return none
 */
void GUTFT_32x32CharacterDelete(char codeUpper, char codeLower){
	unsigned char command[] = {
		0x1f, 0x28, 0x67, 0x15,
		codeUpper, codeLower
	};
	write_data(command, 6);
}

/**
 * Save the download characters defined in RAM to FROM (RAM -> FROM)
 * 
 * fontSize = 0x01: 6x8 pixels
 * fontSize = 0x02: 8x16 pixels
 * fontSize = 0x03: 16x16 pixels
 * fontSize = 0x04: 16x32 pixels
 * fontSize = 0x05: 32x32 pixels
 * fontSize = 0x06: 12x24 pixels
 * 
 * @param  fontSize The desired character font size to be saved.
 * @return none
 */
void GUTFT_downloadCharacterSave(char fontSize){
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x11,
		fontSize
	};
	write_data(command, 5);
}

/**
 * Transfer the downloaded characters saved in FROM to RAM.
 * 
 * fontSize = 0x01: 6x8 pixels
 * fontSize = 0x02: 8x16 pixels
 * fontSize = 0x03: 16x16 pixels
 * fontSize = 0x04: 16x32 pixels
 * fontSize = 0x05: 32x32 pixels
 * fontSize = 0x06: 12x24 pixels
 * 
 * @param  fontSize The desired character font to be restored.
 * @return none
 */
void GUTFT_downloadCharacterRestore(char fontSize){
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x21,
		fontSize
	};
	write_data(command, 5);
}

/**
 * Start user setup mode.
 * 
 * The following data is transmitter when entering user setup mode:
 * 1. Header | 0x28 | 1 byte
 * 2. Identifier 1 | 0x65 | 1 byte
 * 3. Identifier 2 | 0x01 | 1 byte
 * 4. NUL | 0x00 | 1 byte
 * 
 * @return none
 */
void GUTFT_enterUserSetupMode(){
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x01,
		0x49, 0x4e
	};
	write_data(command, 6);
}

/**
 * End user setup mode and initiate a software reset.
 * 
 * @return none
 */
void GUTFT_endUserSetupMode(){
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x02,
		0x4f, 0x55, 0x54
	};
	write_data(command, 7);
}

/**
 * Define the user font table for a specified character size.
 * 
 * table = 0x01: 6x8 pixel user table
 * 	P(0x80 - 1) ... P(0x80 - 6) ... P(0xff - 6)
 * 	6 bytes / font * 128 characters (768 bytes)
 * table = 0x02: 8x16 pixel user table
 * 	P(0x80 - 1) ... P(0x80 - 16) ... P(0xff - 16)
 * 	16 bytes / font * 128 characters (2048 bytes)
 * table = 0x03: 12x24 pixel user table
 * 	P(0x80 - 1) ... P(0x80 - 36) ... P(0xff - 36)
 * 	36 bytes / font * 128 characters (4608 bytes)
 * table - 0x04: 16x32 pixel user table
 * 	P(0x80 - 1) ... P(0x80 - 64) ... P(0xff - 64)
 * 	64 bytes / font * 128 characters (8192 bytes)
 * 
 * It is not possible to only define a part of the character code space.
 * ONLY VALID IN USER SETUP MODE
 * 
 * @param  table The desired user table font size.
 * @param  data  The user table data.
 * @return none
 */
void GUTFT_FROMUserFontDefinition(char table, unsigned char * data){
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x13, table
	};
	write_data(command, 5);
	switch(table){
		case 1:
			write_data(data, 768);
			break;
		case 2:
			write_data(data, 2048);
			break;
		case 3:
			write_data(data, 4608);
			break;
		case 4:
			write_data(data, 8192);
			break;
	}
}

/**
 * Define or delete RAM Macro or RAM Program Macro
 * 
 * 0x0000 <= (lengthLower + lengthUpper * 0x0100) <= 0x0400
 * (lengthLower + lengthUpper * 0x0100) > 0x0000: Supplied data is stored as Macro
 * (lengthLower + lengthUpper * 0x0100) = 0x0000: Macro is deleted
 * 
 * @param  lengthUpper RAM Macro data length, upper byte
 * @param  lengthLower RAM Macro data length, lower byte
 * @param  data        RAM Macro data
 * @returns none
 */
void GUTFT_RAMMacroDefineDelete(char lengthUpper, char lengthLower, unsigned char * data){
	unsigned char command[] = {
		0x1f, 0x3a, lengthLower, lengthUpper
	};
	int length = ((lengthUpper & 0xff) << 8 | lengthLower & 0xff);
	write_data(command, 4);
	write_data(data, length);
}

/**
 * Define or delete FROM Macro or FROM Program Macro
 * 
 * 0x01 <= registrationNum <= 0x04: FROM Macro number 1-4
 * 0x0000 <= (lengthLower + lengthUpper * 0x0100) <= 0x2000 (if using 4 Macros), 0x8000 (if using 1 Macro)
 * 	(lengthLower + lengthUpper * 0x0100) > 0: Supplied data is stored as a Macro
 *  (lengthLower + lengthUpper * 0x0100) = 0: Macro is deleted
 * 0x00 <= interval <= 0xff
 * 0x00 <= idleTime <= 0xff
 * 0x00 <= data <= 0xff
 * 
 * @param  registrationNum FROM Macro registration number
 * @param  lengthUpper     FROM Macro data length, upper byte
 * @param  lengthLower     FROM Macro data length, lower byte
 * @param  interval        Display time interval (interval * IntTime)
 * @param  idleTime        Idle time for macro repetition (idleTime * IntTime)
 * @param  data            FROM Macro data
 * @return none
 */
void GUTFT_FROMMacroDefineDelete(char registrationNum, char lengthUpper, char lengthLower, char interval, char idleTime, unsigned char * data){
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x12,
		registrationNum, lengthLower, lengthUpper, interval,
		idleTime
	};
	int length = ((lengthUpper & 0xff) << 8 | lengthLower & 0xff);
	write_data(command, 9);
	write_data(data, length);
}

/**
 * Continuously execute contents of defined Macro 'definitionNum'
 * 
 * 0x00 <= definitionNum <= 0x04, 0x80 <= definitionNum <= 0x84
 * definitionNum = 0x00: RAM Macro 0
 * 0x01 <= definitionNum <= 0x04: FROM Macro 1-4
 * definitionNum = 0x80: RAM Program Macro 0
 * 0x81 <= definitionNum <= 0x84: FROM Program Macro 1-4
 * 0x00 <= interval <= 0xff
 * 0x00 <= idleTime <= 0xff
 * 
 * @param  definitionNum Macro processing definition number
 * @param  interval      Display time interval (interval * IntTime)
 * @param  idleTime      Idle time for Macro repetition (idleTime * IntTime)
 * @return none
 */
void GUTFT_macroExecution(char definitionNum, char interval, char idleTime){
	unsigned char command[] = {
		0x1f, 0x5e, definitionNum, interval, idleTime
	};
	write_data(command, 5);
}

/**
 * Macro end condition set.
 * 
 * endCodeEnable = 0x00: Macro end code disabled
 * endCodeEnable = 0x01: Macro end code enabled
 * 0x00 <= endCode <= 0xff
 * endScreenSetting = 0x00: Clear screen at Macro end
 * endScreenSetting = 0x01: Do not clear screen at Macro end
 * 
 * @param  endCodeEnable    Macro end code enable/disable
 * @param  endCode          Macro end code
 * @param  endScreenSetting Macro end clear screen setting
 * @return none
 */
void GUTFT_macroEndCondition(char endCodeEnable, uint8_t endCode, uint8_t endScreenSetting){
	unsigned char command[] = {
		0x1f, 0x28, 0x69, 0x20,
		endCodeEnable, endCode, endScreenSetting
	};
	write_data(command, 7);
}

/**
 * Store the supplied data into general-purpose memory / FROM2.
 * 
 * memorySelect = 0x10: FROM2 base address 0x0000.0000
 * memorySelect = 0x11: FROM2 base address 0x0100.0000
 * memorySelect = 0x12: FROM2 base address 0x0200.0000
 * memorySelect = 0x13: FROM2 base address 0x0300.0000
 * memorySelect = 0x14: FROM2 base address 0x0400.0000
 * memorySelect = 0x15: FROM2 base address 0x0500.0000
 * memorySelect = 0x16: FROM2 base address 0x0600.0000
 * memorySelect = 0x17: FROM2 base address 0x0700.0000
 * memorySelect = 0x18: FROM2 base address 0x0800.0000
 * memorySelect = 0x19: FROM2 base address 0x0900.0000
 * memorySelect = 0x1a: FROM2 base address 0x0a00.0000
 * memorySelect = 0x1b: FROM2 base address 0x0b00.0000
 * memorySelect = 0x1c: FROM2 base address 0x0c00.0000
 * memorySelect = 0x1d: FROM2 base address 0x0d00.0000
 * memorySelect = 0x1e: FROM2 base address 0x0e00.0000
 * memorySelect = 0x1f: FROM2 base address 0x0f00.0000
 *  0x000001 <= (sizeLower + sizeUpper * 0x0100 + sizeExtension * 0x00010000) <= 0x00ffffff
 *  0x000000 <= (addressLower + addressUpper * 0x0100 + addressExtension * 0x00010000) <= 0x00ffffff
 * memorySelect = 0x30: General-purpose RAM
 *  0x000001 <= (sizeLower + sizeUpper * 0x0100 + sizeExtension * 0x00010000) <= 0x00000400
 *  0x000000 <= (addressLower + addressUpper * 0x0100 + addressExtension * 0x00010000) <= 0x000003ff
 * memorySelect = 0x31: General-purpose FROM
 *  0x000001 <= (sizeLower + sizeUpper * 0x0100 + sizeExtension * 0x00010000) <= 0x00001000
 *  0x000000 <= (addressLower + addressUpper * 0x0100 + addressExtension * 0x00010000) <= 0x0000ffff
 * 0x00 <= data <= 0xff
 * 
 * @param  sizeUpper        Data size, upper byte
 * @param  sizeLower        Data size, lower byte
 * @param  sizeExtension    Data size, extension byte
 * @param  memorySelect     Desired memory to use
 * @param  addressUpper     Memory address, upper byte
 * @param  addressLower     Memory address, lower byte
 * @param  addressExtension Memory address, extension byte
 * @param  data             Data to store
 * @return none
 */
void GUTFT_memoryStore(unsigned size, uint8_t memorySelect, uint8_t addressUpper, uint8_t addressLower, uint8_t addressExtension, unsigned char * data){
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x18,
		size, size >> 8, size >> 16, memorySelect,
		addressLower, addressUpper, addressExtension
	};
	write_data(command, 11);
	write_data(data, size);
}

/**
 * Transfer data between general-purpose memory / FROM2 areas.
 * 
 * memorySelect = 0x10: FROM2 base address 0x0000.0000
 * memorySelect = 0x11: FROM2 base address 0x0100.0000
 * memorySelect = 0x12: FROM2 base address 0x0200.0000
 * memorySelect = 0x13: FROM2 base address 0x0300.0000
 * memorySelect = 0x14: FROM2 base address 0x0400.0000
 * memorySelect = 0x15: FROM2 base address 0x0500.0000
 * memorySelect = 0x16: FROM2 base address 0x0600.0000
 * memorySelect = 0x17: FROM2 base address 0x0700.0000
 * memorySelect = 0x18: FROM2 base address 0x0800.0000
 * memorySelect = 0x19: FROM2 base address 0x0900.0000
 * memorySelect = 0x1a: FROM2 base address 0x0a00.0000
 * memorySelect = 0x1b: FROM2 base address 0x0b00.0000
 * memorySelect = 0x1c: FROM2 base address 0x0c00.0000
 * memorySelect = 0x1d: FROM2 base address 0x0d00.0000
 * memorySelect = 0x1e: FROM2 base address 0x0e00.0000
 * memorySelect = 0x1f: FROM2 base address 0x0f00.0000
 *  0x000001 <= (sizeLower + sizeUpper * 0x0100 + sizeExtension * 0x00010000) <= 0x00ffffff
 *  0x000000 <= (destAddressLower + destAddressUpper * 0x0100 + destAddressExtension * 0x00010000) <= 0x00ffffff
 *  0x000000 <= (srcAddressLower + srcAddressUpper * 0x0100 + srcAddressExtension * 0x00010000) <= 0x00ffffff
 * memorySelect = 0x30: General-purpose RAM
 *  0x000001 <= (sizeLower + sizeUpper * 0x0100 + sizeExtension * 0x00010000) <= 0x00000400
 *  0x000000 <= (destAddressLower + destAddressUpper * 0x0100 + destAddressExtension * 0x00010000) <= 0x00ffffff
 *  0x000000 <= (srcAddressLower + srcAddressUpper * 0x0100 + srcAddressExtension * 0x00010000) <= 0x000003ff
 * memorySelect = 0x31: General-purpose FROM
 *  0x000001 <= (sizeLower + sizeUpper * 0x0100 + sizeExtension * 0x00010000) <= 0x00001000
 *  0x000000 <= (destAddressLower + destAddressUpper * 0x0100 + destAddressExtension * 0x00010000) <= 0x00ffffff
 *  0x000000 <= (srcAddressLower + srcAddressUpper * 0x0100 + srcAddressExtension * 0x00010000) <= 0x0000ffff
 * 
 * @param  sizeUpper            Transfer data size, upper byte
 * @param  sizeLower            Transfer data size, lower byte
 * @param  sizeExtension        Transfer data size, extension byte
 * @param  destMemory           Destination memory select
 * @param  destAddressUpper     Destination address, upper byte
 * @param  destAddressLower     Destination address, lower byte
 * @param  destAddressExtension Destination address, extension byte
 * @param  srcMemory            Source memory select
 * @param  srcAddressUpper      Source address, upper byte
 * @param  srcAddressLower      Source address, lower byte
 * @param  srcAddressExtension  Source address, extension byte
 * @return none
 */
void GUTFT_memoryTransfer(uint8_t sizeUpper, uint8_t sizeLower, uint8_t sizeExtension, uint8_t destMemory, uint8_t destAddressUpper, uint8_t destAddressLower, uint8_t destAddressExtension, uint8_t srcMemory, uint8_t srcAddressUpper, uint8_t srcAddressLower, uint8_t srcAddressExtension){
	unsigned char command[] = {
		0x1f, 0x28, 0x65, 0x19,
		sizeLower, sizeUpper, sizeExtension,
		destMemory, destAddressLower, destAddressUpper, destAddressExtension,
		srcMemory, srcAddressLower, destAddressUpper, srcAddressExtension
	};
}
