/*********************************************************
 * Purpose:
 * GT-CP USB wrapper library using libusb. 
 * 
 * Version: 1.0
 * Website: noritake-elec.com
 *********************************************************/

#include <libusb-1.0/libusb.h>
#include "GUTFT_USB_LIB.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

// Vendor ID value for Noritake
#define VENDOR_ID 0x0eda
// Product ID value for GT-CP (7" and 4.3")
#define PRODCT_ID 0x1200

// USB Device Handle
static struct libusb_device_handle *devh = NULL;

// IN and OUT endpoint addresses
static int ep_in_addr = 0x81;
static int ep_out_addr = 0x02;

/**
 * Start the USB interface and establish a connection with
 * a GT-CP module.
 * 
 * @returns int Operation result. 0 is OK, 1 is FAIL.
 */
int USB_start(){
	int rc;
	int result;
	int retVal;

	rc = libusb_init(NULL);
	if(rc < 0){
		fprintf(stderr, "Error initializing libusb: %s\n", libusb_error_name(rc));
		exit(1);
	}

	devh = libusb_open_device_with_vid_pid(NULL, VENDOR_ID, PRODCT_ID);
	if(!devh){
		fprintf(stderr, "Error finding USB device.\n");
		fprintf(stderr, "Make sure your module is connected via USB.\n");
		if(devh){
			libusb_close(devh);
		}
		libusb_exit(NULL);
		retVal = 1;
	} else{
		result = libusb_claim_interface(devh, 0);
		if(result >= 0)
		{
			retVal = 0;
		} else {
			fprintf(stderr, "USB interface claim error %d\n", result);
			if(devh){
				libusb_close(devh);
			}
			libusb_exit(NULL);
			retVal = 1;
		}
	}
	return retVal;
}

/**
 * Writes a single character to the USB device.
 * 
 * @param   c    The byte to be written.
 * @returns none
 */
void write_char(unsigned char c){
	int actual_length;
	if(libusb_bulk_transfer(devh, ep_out_addr, &c, 1, &actual_length, 0) < 0){
		fprintf(stderr, "Error while sending char\n");
	}
}

/**
 * Writes multiple characters to the USB device.
 * 
 * @param   data The characters to be written.
 * @returns none
 */
void write_multi_char(unsigned char * data){
	int actual_length;
	if(libusb_bulk_transfer(devh, ep_out_addr, data, strlen(data), &actual_length, 0) < 0){
		fprintf(stderr, "Error while sending char\n");
	}
}

/**
 * Writes a specified data set to the USB device.
 * 
 * @param   data The data set to be written.
 * @param   size The number of bytes to be written.
 * @returns none
 */
void write_data(unsigned char * data, int size){
	int actual_length;
	if(libusb_bulk_transfer(devh, ep_out_addr, data, size, &actual_length, 0) < 0){
		fprintf(stderr, "Error while sending char\n");
	}
}

/**
 * Read data over USB.
 *
 * @param  data The data array to store the read data.
 * @param  size The number of bytes to read.
 * @return Number of bytes received.
 */
int read_USB(unsigned char * data, int size){
	int actual_length;
	if(libusb_bulk_transfer(devh, ep_in_addr, data, size, &actual_length, 1000) < 0){
		fprintf(stderr, "Error while reading data from USB.\n");
	}
	printf("Received: %d\n", actual_length);
	return actual_length;
}

/**
 * This function will release the claimed interface,
 * close the device handler, and deinitialize libusb.
 *
 * @returns none
 */
void release_USB(){
	libusb_release_interface(devh, 0);
	libusb_close(devh);
	libusb_exit(NULL);
}
